
/*

	Save Render Log File
	Matt Gorner
	17 October 2009
	
	www.pixsim.co.uk

	Saves scene render settings information and render times to a chosen text file.
	
	How to use:

	Apply as a Master Script, open 'Master Plugins' (CTRL q) pull down on 'Save Render Log File'
	
	Double-click on 'Save Render Log File' and set the file save name
	When first applied it automatically sets the log filename to the scene file name with '_RENDER_LOG.txt' appended.
	The path defaults to the scene file location.

	Also, there's an option to format the .txt file with vbulletin board codes for nicer looking posts :)

*/


@version 2.2
@warnings
@script master
@name "Save Render Log File"

@define SCRIPT_VERSION "1.0"

// Global Variables

host;
formatvb = 0;
frame_rendering;
aborted = FALSE;
logfilepath, logfilename, logfileextension;

hrsRenderBegin,minRenderBegin,secRenderBegin,timeRenderBegin;
hrsFrameStart,minFrameStart,secFrameStart,timeFrameStart;
hrsFrameEnd,minFrameEnd,secFrameEnd,timeFrameEnd;

sceneflagnames;
sceneflagvalues;
numberofcameras = 0;
currentcamera = 0;
cameratype = "";

// An array of the scene item flag names we are interested in

sceneflagnames = @
	"ResolutionMultiplier",
	"FrameSize",
	"UseGlobalResolution",
	"GlobalFrameSize",
	"GlobalMultiplier",
	"Plugin CameraHandler",
	"EnableRadiosity",
	"RadiosityType",
	"RadiosityInterpolated",
	"BlurRadiosity",
	"RadiosityTransparency",
	"VolumetricRadiosity",
	"RadiosityUseAmbient",
	"RadiosityDirectionalRays",
	"RadiosityUseGradients",
	"RadiosityUseBehindTest",
	"RadiosityFlags",
	"RadiosityIntensity",
	"IndirectBounces",
	"RadiosityRays",
	"SecondaryBounceRays",
	"RadiosityTolerance",
	"RadiosityMinPixelSpacing",
	"RadiosityMaxPixelSpacing",
	"RadiosityMultiplier",
	"CacheRadiosity",
	"PreprocessRadiosity",
	"RadiosityCacheAnimation",
	"RadiosityCacheModulus",
	"AASamples",
	"Sampler",
	"AdaptiveThreshold",
	"Oversampling",
	"Antialiasing",
	"EnhancedAA",
	"AntiAliasingLevel",
	"ReconstructionFilter",
	"AdaptiveSampling",
	"RayRecursionLimit",
	"RayPrecision",
	"RayCutoff",
	"CurrentCamera"@;


// One-time initialization takes place here

create
{
	// Init variables

	host = runningUnder();

	s = split(Scene().filename);

	logfilepath = s[1] + s[2];
	logfileextension = ".txt";
	logfilename = s[3] + "_RENDER_LOG";

	// Open the script interface when applied as a Master Script
    // options();
}


flags
{
    // Tell LightWave what type of master script level we are:
    // LWMASTF_LAYOUT	= Layout level script, these aren't removed if the scene is cleared
	// LWMAST_SCENE		= Scene level script, these will be removed if the scene is cleared

    // Also tell it we want to recieve notifications:
    // LWMASTF_RECEIVE_NOTIFICATIONS

    // These flags definitions can be found in the LW SDK Include file: 'lwmaster.h'

    return(SCENE,LWMASTF_RECEIVE_NOTIFICATIONS);
    // return(LWMASTF_LAYOUT, LWMASTF_RECEIVE_NOTIFICATIONS);
}


options
{
	// Is the requester open already?  Then close it.
    if(reqisopen())
        reqend();
    else
    {
    	// Otherwise open it up ...
    
	    // User Interface Layout Variables (this is me being anal!) :D

		ui_spacing			= 3;										// Spacing gap size

		gad_x				= 10;										// Gadget X coord
		gad_y				= 24;										// Gadget Y coord
		gad_w				= 650;										// Gadget width
		gad_h				= 19;										// Gadget height
		gad_text_offset		= 120;										// Gadget text offset
		num_gads			= 2;										// Total number of gadgets vertically (for calculating the max window height)
		ui_offset_x 		= 0;										// Main X offset from 0
		ui_offset_y 		= ui_spacing*2;								// Main Y offset from 0
		ui_row				= 0;										// Row number
		ui_row_offset		= gad_h + ui_spacing;						// Row offset
		ui_window_w			= 700;										// Window width
		ui_window_h			= ui_row_offset*num_gads+(4*ui_spacing);	// Window height

		reqbegin("Save Render Log File ...");
		reqsize(ui_window_w,ui_window_h);

		gad_savefile = ctlfilename("Render Log File:", logfilepath + logfilename + logfileextension,,0);
		ctlposition(gad_savefile, gad_x, ui_offset_y + 2, gad_w, gad_h, gad_text_offset);

    	ui_offset_y += ui_row_offset;

		gad_formatvb = ctlcheckbox("Format with vBulletin Forum Codes?", formatvb);
		ctlposition(gad_formatvb, gad_x, ui_offset_y + ui_spacing, 200 + gad_text_offset, gad_h, gad_text_offset);

		// Got to these callback procedures when gadgets have changed to update the variables globally
		ctlrefresh( gad_savefile, "update_save_file_properties");
		ctlrefresh( gad_formatvb, "update_format_properties");

		// Open the script UI
    	reqopen();
    }
}


// Gadget refresh callback procedures ...

update_save_file_properties: value
{
	// Get the path and split into various parts and store globally
	s = split(value);
	logfilepath = s[1] + s[2];
	logfilename = s[3];
	logfileextension = s[4];
}

update_format_properties: value
{
	// Store the vbulletin codes option globally
	formatvb = value;
}


// Save procedure for when the scene is saved (store our scripts settings)

save: mode, io
{
	// Don't do any saving if we're not in SCENEMODE
    return if mode == OBJECTMODE;

	// Write the variables we need to the scene file
    io.writeln(logfilepath);
    io.writeln(logfilename);
    io.writeln(logfileextension);
    io.writeln(formatvb);
}

// Load procedure for when the scene is loaded (grab the variables that were stored if they are present)

load: mode,io
{
	// Don't do any loading if we're not in SCENEMODE
    return if mode == OBJECTMODE;

	// Read the variables back in
	logfilepath = io.read();
	logfilename = io.read();
	logfileextension = io.read();
	formatvb = io.read().asNum();
}


// LightWave events handling

process: event, command
{
	// Don't do for screamernet, most people will be using a render controller which stores this info anyway.
    if(host == SCREAMERNET)
        return;

	// We were passed an event - render has been aborted by the user.
    if(event == LWEVNT_NOTIFY_RENDER_ABORTED)
    {
    	aborted = TRUE;
    }

	// We were passed an event - render has started
	// Is passed whether F9 or F10
    if(event == LWEVNT_NOTIFY_RENDER_STARTING)
    {
    	aborted = FALSE;
    	
    	// Store when the render began ...
        (hrsRenderBegin,minRenderBegin,secRenderBegin,timeRenderBegin) = time();

        // Get the date variables
		(d,m,y,w,j,sm,sw) = date();

		// Format the time into something more presentable
        if(hrsRenderBegin >= 12)
        {
        	if(hrsRenderBegin == 12)
        	{
        		h = string(hrsRenderBegin);
        	}else
        	{
        		h = "0" + string(hrsRenderBegin - 12);
        	}
        	tag = "pm";
        }else
        {
        	if(hrsRenderBegin >9)
        	{
        		h = string(hrsRenderBegin);
        	}else
        	{
        		h = "0" + string(hrsRenderBegin);
        	}
           	tag = "am";
        }

        if(minRenderBegin < 10)
        {
        	m = "0" + string(minRenderBegin);
        }else
        {
        	m = string(minRenderBegin);
        }

        if(secRenderBegin < 10)
        {
        	s = "0" + string(secRenderBegin);
        }else
        {
        	s = string(secRenderBegin);
        }
        
		// Get the path and name of the currently loaded scene file, and seperate out the path from the name
		scene = split(Scene().filename);

		// Re-construct the scene name and path separately, we are doing this because there is no Scene().path command
		spath = scene[1] + scene[2];
		sname = scene[3] + scene[4];

		// Open the log file ...
		log_file_handle = File(logfilepath + logfilename + logfileextension,"w") || error("Cannot open log file" + logfilepath + logfilename + logfileextension + " for writing");
		
		// Write the header stuff ...
		log_file_handle.writeln("-------------------------------------------------------------");
		if ( formatvb )
		{
			log_file_handle.writeln("[B]Render Log File[/B]");
			log_file_handle.writeln("-------------------------------------------------------------");
			log_file_handle.writeln("[SIZE=\"1\"][B]Creation Date:[/B] " + sw + " " + d + " " + sm + " " + y );
			log_file_handle.writeln("[B]Creation Time:[/B] " + h + ":" + m + ":" + s + tag);
			log_file_handle.writeln("[B]Scene File:[/B] " + sname);
			log_file_handle.writeln("[B]Scene File Path:[/B] " + spath + "[/SIZE]");
			log_file_handle.writeln("");
			log_file_handle.writeln("-------------------------------------------------------------");
			log_file_handle.writeln("[B]Scene Render Settings[/B]");
			log_file_handle.writeln("-------------------------------------------------------------");
		}else
		{
			log_file_handle.writeln("Render Log File");
			log_file_handle.writeln("-------------------------------------------------------------");
			log_file_handle.writeln("Creation Date: " + sw + " " + d + " " + sm + " " + y );
			log_file_handle.writeln("Creation Time: " + h + ":" + m + ":" + s + tag);
			log_file_handle.writeln("Scene File: " + sname);
			log_file_handle.writeln("Scene File Path: " + spath);
			log_file_handle.writeln("");
			log_file_handle.writeln("-------------------------------------------------------------");
			log_file_handle.writeln("Scene Render Settings");
			log_file_handle.writeln("-------------------------------------------------------------");
		}
		log_file_handle.close();

		// Get the render info for the scene file currently opened (but grab the info from the file ON DISK)
		// We have to do this because there are loads of render settings not available to LScript yet
		// NOTE: The procedure opens and closes any files passed to it
		get_scene_file_information( Scene().filename, logfilepath + logfilename + logfileextension );

		// Open the log file again, but in append mode ...
		log_file_handle = File(logfilepath + logfilename + logfileextension,"a") || error("Cannot open log file" + logfilepath + logfilename + logfileextension + " for writing");

		// Write out the frame render times titles ...
		if ( formatvb )
		{
		log_file_handle.writeln("");
		log_file_handle.writeln("-------------------------------------------------------------");
		log_file_handle.writeln("[B]Individual Frame Render Times[/B]");
		log_file_handle.writeln("-------------------------------------------------------------");
		}else
		{
		log_file_handle.writeln("");
		log_file_handle.writeln("-------------------------------------------------------------");
		log_file_handle.writeln("Individual Frame Render Times");
		log_file_handle.writeln("-------------------------------------------------------------");
		}
		log_file_handle.close();
    }
    

	// We were passed an event - a frame has started rendering
	// Is passed whether F9 or F10
    if(event == LWEVNT_NOTIFY_RENDER_FRAME_STARTING)
    {
    	// Reset frame render start times
        (hrsFrameStart,minFrameStart,secFrameStart,timeFrameStart) = time();
    }


	// We were passed an event - frame has finished rendering
	// Is passed whether F9 or F10, but also annoyingly if the render is aborted, grrrrrrrr! (LW SDK bug!)
    if(event == LWEVNT_NOTIFY_RENDER_FRAME_COMPLETE )
    {
		// Get the frame we are on, amazingly, there is no variable for this in LScript! WTF!
		// So we need to do this ...
        frame_rendering = round(Scene().fps * Scene().currenttime,1);
        
		// Get time the frame finished rendering ...
    	(hrsFrameEnd,minFrameEnd,secFrameEnd,timeFrameEnd) = time();

		// Format the time into something more presentable
        if(hrsFrameEnd >= 12)
        {
        	if(hrsFrameEnd == 12)
        	{
        		end_h = string(hrsFrameEnd);
        	}else
        	{
        		end_h = "0" + string(hrsFrameEnd - 12);
        	}
        	end_tag = "pm";
        }else
        {
        	if(hrsFrameEnd >9)
        	{
        		end_h = string(hrsFrameEnd);
        	}else
        	{
        		end_h = "0" + string(hrsFrameEnd);
        	}
           	end_tag = "am";
        }

        if(minFrameEnd < 10)
        {
        	end_m = "0" + string(minFrameEnd);
        }else
        {
        	end_m = string(minFrameEnd);
        }

        if(secFrameEnd < 10)
        {
        	end_s = "0" + string(secFrameEnd);
        }else
        {
        	end_s = string(secFrameEnd);
        }

		// Format the time into something more presentable
        if(hrsFrameStart >= 12)
        {
        	if(hrsFrameStart == 12)
        	{
        		start_h = string(hrsFrameStart);
        	}else
        	{
        		start_h = "0" + string(hrsFrameStart - 12);
        	}
        	start_tag = "pm";
        }else
        {
        	if(hrsFrameStart >9)
        	{
        		start_h = string(hrsFrameStart);
        	}else
        	{
        		start_h = "0" + string(hrsFrameStart);
        	}
           	start_tag = "am";
        }

        if(minFrameStart < 10)
        {
        	start_m = "0" + string(minFrameStart);
        }else
        {
        	start_m = string(minFrameStart);
        }

        if(secFrameStart < 10)
        {
        	start_s = "0" + string(secFrameStart);
        }else
        {
        	start_s = string(secFrameStart);
        }

		// Open the log file again in append mode ...
		log_file_handle = File(logfilepath + logfilename + logfileextension,"a") || error("Cannot open log file" + logfilepath + logfilename + logfileextension + " for writing");

		// Work out the frame render time
		t = string(convert_time( max(timeFrameStart,timeFrameEnd) - min(timeFrameStart,timeFrameEnd)));

		// This abort thing didn't work, I can't seem to get this script to stop saving frame info if the render is aborted, I blame LScript! ;)
		// if (aborted == FALSE)
		// {

		// Write out the frame render times ...
		if ( formatvb )
		{
			log_file_handle.writeln( "[SIZE=\"1\"][B]Frame:[/B] " + frame_rendering);
			log_file_handle.writeln( "[B]Started:[/B] " + start_h + ":" + start_m + ":" + start_s + start_tag);
			log_file_handle.writeln( "[B]Completed:[/B] " + end_h + ":" + end_m + ":" + end_s + end_tag);
			log_file_handle.writeln( "[B]Render Time:[/B] " + t + "[/SIZE]" );
	
			// This abort check doesn't work, but I've left it here to prove I tried! :D
			if(aborted)
			{
				log_file_handle.writeln( "[B]Render was ABORTED![/B]" );
			}
			log_file_handle.writeln("");
		}else
		{
			log_file_handle.writeln( "Frame: " + frame_rendering);
			log_file_handle.writeln( "Started: " + start_h + ":" + start_m + ":" + start_s + start_tag);
			log_file_handle.writeln( "Completed: " + end_h + ":" + end_m + ":" + end_s + end_tag);
			log_file_handle.writeln( "Render Time: " + t );
			
			// This abort check doesn't work, but I've left it here to prove I tried! :D
			if(aborted)
			{
				log_file_handle.writeln( "Render was ABORTED!" );
			}
			log_file_handle.writeln("");
		}
		log_file_handle.close();
		
		// } // (aborted == FALSE)


		// Are we rendering the last frame?  If so, then add total render time info
		// Have to do this because of issues with 'LWEVNT_NOTIFY_RENDER_COMPLETE'

        if(frame_rendering == Scene().renderend)
        {

		// The following commented code is no longer needed, as it's already in this event code,
		// see the 'LWEVNT_NOTIFY_RENDER_COMPLETE' code below as to why.

		// Get time the frame finished rendering ...
    	// (hrsFrameEnd,minFrameEnd,secFrameEnd,timeFrameEnd) = time();

		// Format the time into something more presentable
		/*
        if(hrsFrameEnd >= 12)
        {
        	if(hrsFrameEnd == 12)
        	{
        		end_h = string(hrsFrameEnd);
        	}else
        	{
        		end_h = "0" + string(hrsFrameEnd - 12);
        	}
        	end_tag = "pm";
        }else
        {
        	if(hrsFrameEnd >9)
        	{
        		end_h = string(hrsFrameEnd);
        	}else
        	{
        		end_h = "0" + string(hrsFrameEnd);
        	}
           	end_tag = "am";
        }

        if(minFrameEnd < 10)
        {
        	end_m = "0" + string(minFrameEnd);
        }else
        {
        	end_m = string(minFrameEnd);
        }

        if(secFrameEnd < 10)
        {
        	end_s = "0" + string(secFrameEnd);
        }else
        {
        	end_s = string(secFrameEnd);
        }

		*/
        
		// Format the time into something more presentable
		if(hrsRenderBegin >= 12)
        {
        	if(hrsRenderBegin == 12)
        	{
        		begin_h = string(hrsRenderBegin);
        	}else
        	{
        		begin_h = "0" + string(hrsRenderBegin - 12);
        	}
        	begin_tag = "pm";
        }else
        {
        	if(hrsRenderBegin >9)
        	{
        		begin_h = string(hrsRenderBegin);
        	}else
        	{
        		begin_h = "0" + string(hrsRenderBegin);
        	}
           	begin_tag = "am";
        }

        if(minRenderBegin < 10)
        {
        	begin_m = "0" + string(minRenderBegin);
        }else
        {
        	begin_m = string(minRenderBegin);
        }

        if(secRenderBegin < 10)
        {
        	begin_s = "0" + string(secRenderBegin);
        }else
        {
        	begin_s = string(secRenderBegin);
        }

		// Open log file in append mode
		log_file_handle = File(logfilepath + logfilename + logfileextension,"a") || error("Cannot open log file" + logfilepath + logfilename + logfileextension + " for writing");
		
		// Work out the total render time
		t = string(convert_time( max(timeRenderBegin,timeFrameEnd) - min(timeRenderBegin,timeFrameEnd)));

		// Write the total render time footer ...
		if ( formatvb )
		{
			log_file_handle.writeln("-------------------------------------------------------------");
			log_file_handle.writeln("[B]Total Render Time[/B]");
			log_file_handle.writeln("-------------------------------------------------------------");
			log_file_handle.writeln( "[B][SIZE=\"1\"]Started:[/B] " + begin_h + ":" + begin_m + ":" + begin_s + begin_tag);
			log_file_handle.writeln( "[B]Finished:[/B] " + end_h + ":" + end_m + ":" + end_s + end_tag);
			log_file_handle.writeln( "[B]Total Render Time:[/B] " + t + "[/SIZE]" );
		}else
		{
			log_file_handle.writeln("-------------------------------------------------------------");
			log_file_handle.writeln("Total Render Time");
			log_file_handle.writeln("-------------------------------------------------------------");
			log_file_handle.writeln( "Started: " + begin_h + ":" + begin_m + ":" + begin_s + begin_tag);
			log_file_handle.writeln( "Completed: " + end_h + ":" + end_m + ":" + end_s + end_tag);
			log_file_handle.writeln( "Total Render Time: " + t );
		}
		log_file_handle.close();

        }

    }


	// We were passed an event - rendering is complete
	// Is passed when ALL frames are done, not inbetween (but annoyingly only AFTER the 'Render Status' window is closed)
    if(event == LWEVNT_NOTIFY_RENDER_COMPLETE)
    {
    	/*
		
		Had to remove the 'total render time' code from this event because the event
		only gets passed to the script when the user closes the 'Render Status' window.
		
		Which means when rendering a sequence of frames, if the user closes the window later
		that when the render _actually_ finished, getting timings based on this event would be
		totally incorrect.
		
		Instead, this code has been shoved into the 'LWEVNT_NOTIFY_RENDER_FRAME_COMPLETE' event.
		But has a check to see if it's rendering the last frame, fortunately this works for F9 renders
		as well as F10 ones, otherwise it would have been a total ball ache to do properly!

		*/

		// Get time the frame finished rendering ...
    	// (hrsFrameEnd,minFrameEnd,secFrameEnd,timeFrameEnd) = time();

		// Format the time into something more presentable
		/*
        if(hrsFrameEnd >= 12)
        {
        	if(hrsFrameEnd == 12)
        	{
        		end_h = string(hrsFrameEnd);
        	}else
        	{
        		end_h = "0" + string(hrsFrameEnd - 12);
        	}
        	end_tag = "pm";
        }else
        {
        	if(hrsFrameEnd >9)
        	{
        		end_h = string(hrsFrameEnd);
        	}else
        	{
        		end_h = "0" + string(hrsFrameEnd);
        	}
           	end_tag = "am";
        }

        if(minFrameEnd < 10)
        {
        	end_m = "0" + string(minFrameEnd);
        }else
        {
        	end_m = string(minFrameEnd);
        }

        if(secFrameEnd < 10)
        {
        	end_s = "0" + string(secFrameEnd);
        }else
        {
        	end_s = string(secFrameEnd);
        }

		*/
        
		// Format the time into something more presentable
		/*
		if(hrsRenderBegin >= 12)
        {
        	if(hrsRenderBegin == 12)
        	{
        		begin_h = string(hrsRenderBegin);
        	}else
        	{
        		begin_h = "0" + string(hrsRenderBegin - 12);
        	}
        	begin_tag = "pm";
        }else
        {
        	if(hrsRenderBegin >9)
        	{
        		begin_h = string(hrsRenderBegin);
        	}else
        	{
        		begin_h = "0" + string(hrsRenderBegin);
        	}
           	begin_tag = "am";
        }

        if(minRenderBegin < 10)
        {
        	begin_m = "0" + string(minRenderBegin);
        }else
        {
        	begin_m = string(minRenderBegin);
        }

        if(secRenderBegin < 10)
        {
        	begin_s = "0" + string(secRenderBegin);
        }else
        {
        	begin_s = string(secRenderBegin);
        }
		*/
		// Open log file in append mode
		// log_file_handle = File(logfilepath + logfilename + logfileextension,"a") || error("Cannot open log file" + logfilepath + logfilename + logfileextension + " for writing");

		// Work out the total render time
		// t = string(convert_time( max(timeRenderBegin,timeFrameEnd) - min(timeRenderBegin,timeFrameEnd)));

		// Write the total render time footer ...
		/*
		if ( formatvb )
		{
			log_file_handle.writeln("-------------------------------------------------------------");
			log_file_handle.writeln("[B]Total Render Time[/B]");
			log_file_handle.writeln("-------------------------------------------------------------");
			log_file_handle.writeln( "[B][SIZE=\"1\"]Started:[/B] " + begin_h + ":" + begin_m + ":" + begin_s + begin_tag);
			log_file_handle.writeln( "[B]Finished:[/B] " + end_h + ":" + end_m + ":" + end_s + end_tag);
			log_file_handle.writeln( "[B]Total Render Time:[/B] " + t + "[/SIZE]" );
		}else
		{
			log_file_handle.writeln("-------------------------------------------------------------");
			log_file_handle.writeln("Total Render Time");
			log_file_handle.writeln("-------------------------------------------------------------");
			log_file_handle.writeln( "Started: " + begin_h + ":" + begin_m + ":" + begin_s + begin_tag);
			log_file_handle.writeln( "Completed: " + end_h + ":" + end_m + ":" + end_s + end_tag);
			log_file_handle.writeln( "Total Render Time: " + t );
		}
		log_file_handle.close();
		*/
    }

}


// Procedure to format the render times into English!

convert_time: value
{
	// For testing time conversion routines without needing to wait for renders
	// hh = 48;
	// mm = 32;
	// ss = 45;
	// value = (60 * 60 * hh) + (60 * mm) + ss;

	// Only took seconds, go Mark Granger's rendering code! :D
 	if ( value <= 60 )
 	{
 		if (value == 60)
 		{
 			render_time = "1 Minute";
 		}else
 		{
 			render_time = value.asStr() + " Seconds";
 		}
 	}

	// Took minutes ...
 	if ( value > 60 && value < 3600 )
 	{
 		// How many minutes?
		m = integer(string( value / 60 ));
		
		// How many seconds?
 		s = m % 60;

 		render_time = m.asStr() + " Min " + s.asStr() + " Sec";
 	}

	// Took hours ...
 	if ( value >= 3600 )
 	{
 		// How many hours?
		h = integer(string( (value / 60) / 60 ));

		// How many minutes?
		m = integer(string( value / 60 ));
		m = m % 60;

		// How many seconds?
 		s = value % 60;

 		render_time = h.asStr() + " Hr " + m.asStr() + " Min " + s.asStr() + " Sec";
 	}

	// Pass back the render time all formatted ...
	return render_time;
}

// Extract scene flag value

getcameravalue: flagname, cameranumber
{
    temp = 0;

	// Go through all names in the scene flags array
	for(loop = 1; loop < sizeof(sceneflagvalues) + 1; loop++)
	{
		// string_to_check = sceneflagvalues[loop];

		if(strleft(sceneflagvalues[loop], size(flagname) ) == flagname )
		{
            if( temp == cameranumber)
            {
                found = loop;
                break;
            } else
            {
                // No such scene flag, return -1
                found = -1;
            }

            temp ++;
		}
	}

    // Pass back the entries location in the sceneflag array
    return( found.asNum() );
}




// Extract scene flag value

getsceneflagvalue: flagname
{
	// Go through all names in the scene flags array
	for(loop = 1; loop <= sizeof(sceneflagvalues); loop++)
	{
		// string_to_check = sceneflagvalues[loop];

		if(strleft(sceneflagvalues[loop], size(flagname) ) == flagname )
		{
			found = loop; // Store where we found the entry in the sceneflag array
			break; // Break out of the loop
		}
		else
		{
		      // No such scene flag, return -1
		      found = -1;
		}
	}

    // Pass back the entries location in the sceneflag array
    return( found.asNum() );
}

// Big scene file render settings scanning procedure!
// This also writes to the log file passed to it

get_scene_file_information: path_of_scenefile_to_scan, path_of_file_to_write_to
{

 	// Open scene file for reading ...
	scenefile = File(path_of_scenefile_to_scan,"r") || error("Cannot open scene file" + path_of_scenefile_to_scan);

	// Get the scene version number (remove spaces at start and end, just in case)
	line = scenefile.read();
	line = scenefile.read();
    line.pack();
    line.trunc();

	// Not a 9.6 scene file?  Then gracefully exit.
	// This script currently only works on LightWave v9.6 and above Scene files
	// But we can still save the render times, so don't stop the script completely
    if(line < "5" )
    {
		log_file_handle = File(path_of_file_to_write_to,"a") || error("Cannot open log file" + path_of_file_to_write_to + " for writing");
		log_file_handle.writeln( "Error: Scene file was not a LightWave 9.6");
		log_file_handle.writeln( "This script can only process 9.6 scene files");
		log_file_handle.close();
    	scenefile.close();
    	return;
    }

    // Count the number of lines in the scene file
	numberoflines = scenefile.linecount();

	// Open log file writing ...
	log_file_handle = File(path_of_file_to_write_to,"a") || error("Cannot open log file" + path_of_file_to_write_to + " for writing");


    // Gather all sceneflag settings and store in sceneflagvalues array, loop until scene file end of file is reached
	while( !scenefile.eof() )
	{
        // Loop through all scene flag names we are interested in
		for(loop = 1; loop <= sizeof(sceneflagnames); loop++)
		{
			// Make sure line read in STARTS with the name we are looking for, helps avoid errors
			if(strleft(line, size(sceneflagnames[loop]) ) == sceneflagnames[loop] )
			{
				// Store line from scene file into array
				sceneflagvalues += line;

				// Found our name, so might as well break out of the current loop for the next one
				break;
			}
		}

		// Read next line of scene file (and remove spaces at start and end, just in case)
		line = scenefile.read();
		line.pack();
        line.trunc();
	}

	// Done with scene file, close it.
    scenefile.close();




    // Write camera settings to file ...
    if ( formatvb ) log_file_handle.writeln("[SIZE=\"1\"][B]Camera Settings:[/B]");
        else log_file_handle.writeln("Camera Settings:");

     // Find out how many cameras are in the scene
	for(loop = 1; loop < sizeof(sceneflagvalues) + 1; loop++)
	{
		// string_to_check = sceneflagvalues[loop];

        // Found a camera?
		if(strleft(sceneflagvalues[loop], size("Plugin CameraHandler") ) == "Plugin CameraHandler" )
        {
            // Yes, add to number of cameras found in scene
            numberofcameras ++;
        }
	}

    // Find active camera
    if( getsceneflagvalue("CurrentCamera") != -1 )
    {
        location_in_array = getsceneflagvalue("CurrentCamera");
        array = parse(" ", sceneflagvalues[location_in_array]);
        currentcamera = array[2].asNum();

        // log_file_handle.writeln( "Current Camera: " + currentcamera );
    }


    // Get Camera Type
    location_in_array = getcameravalue("Plugin CameraHandler", currentcamera);
    array = parse(" ", sceneflagvalues[location_in_array]);

    // Get camera type from array
    cameratype = array[4].asStr();

    // Scene file name tweak!
    if(cameratype == "RealLens")
    {
        cameratype = "Real Lens";
    }
    if(cameratype == "ShiftCamera")
    {
        cameratype = "Shift";
    }
    if(cameratype == "surfaceBaking")
    {
        cameratype = "Surface Baking";
    }
    log_file_handle.writeln( "Camera Type: " + cameratype );

    global = FALSE;

    // Does the current camera use Global Settings?
    location_in_array = getcameravalue("UseGlobalResolution", currentcamera);
    array = parse(" ", sceneflagvalues[location_in_array]);

    if( array[2].asNum() == 0)
    {
        // No it doesn't grab the normal values
        // Get Camera Frame Size
        location_in_array = getcameravalue("FrameSize", currentcamera);
        array = parse(" ", sceneflagvalues[location_in_array]);
        log_file_handle.writeln( "Frame Size: " + array[2].asNum() + " x " + array[3].asNum() );
    
    
        // Get Camera Multiplier
        location_in_array = getcameravalue("ResolutionMultiplier", currentcamera);
        array = parse(" ", sceneflagvalues[location_in_array]);
        log_file_handle.writeln( "Multiplier: " + array[2].asNum() * 100 + "%");
    } else
    {
        // Yes it does, grab the Global values
        global = TRUE;

        // Get Global Frame Size
        if( getsceneflagvalue("GlobalFrameSize") != -1 )
        {
            location_in_array = getsceneflagvalue("GlobalFrameSize");
            array = parse(" ", sceneflagvalues[location_in_array]);

            log_file_handle.writeln( "Frame Size: " + array[2].asNum() + " x " + array[3].asNum() );
        } 


        // Get Global Multiplier
        if( getsceneflagvalue("GlobalMultiplier") != -1 )
        {
            location_in_array = getsceneflagvalue("GlobalMultiplier");
            array = parse(" ", sceneflagvalues[location_in_array]);

            log_file_handle.writeln( "Multiplier: " + array[2].asNum() * 100 + "%" );
        }
    } // Does the current camera use Global Settings?






    // Write antialiasing settings to file ...

    log_file_handle.writeln(""); // Newline
    if ( formatvb ) log_file_handle.writeln("[B]Antialiasing Settings:[/B]");
        else log_file_handle.writeln("Antialiasing Settings:");


    // Are we using a Classic camera?
   	if( cameratype != "Classic" )
   	{
   	    // Get non-classic camera AA settings

        // Antialiasing Setting
        if( getsceneflagvalue("AASamples") != -1 )
        {
            location_in_array = getsceneflagvalue("AASamples");
            array = parse(" ", sceneflagvalues[location_in_array]);

            log_file_handle.writeln( "Antialiasing: " + array[2].asNum());
        }


        // Reconstruction Filter
        if( getsceneflagvalue("ReconstructionFilter") != -1 )
        {
            location_in_array = getsceneflagvalue("ReconstructionFilter");
            array = parse(" ", sceneflagvalues[location_in_array]);

            str = "Reconstruction Filter: ";
    
            // Convert sampling type number to corresponding name in UI
            switch( array[2].asNum() )
        	   {
					case 0:
                        str = str + "Classic";
                        break;
        			case 1:
                        str = str + "Box";
                        break;
        			case 3:
        				str = str + "Box (Soft)";
        				break;
        			case 2:
        				str = str + "Box (Sharp)";
        				break;
        			case 4:
        				str = str + "Gaussian";
        				break;
        			case 6:
        				str = str + "Gaussian (Soft)";
        				break;
        			case 5:
        				str = str + "Gaussian (Sharp)";
        				break;
        			case 7:
        				str = str + "Mitchell";
        				break;
        			case 9:
        				str = str + "Mitchell (Soft)";
        				break;
        			case 8:
        				str = str + "Mitchell (Sharp)";
        				break;
        			case 10:
        				str = str + "Lanczos";
        				break;
        			case 12:
        				str = str + "Lanczos (Soft)";
        				break;
        			case 11:
        				str = str + "Lanczos (Sharp)";
        				break;
                } // switch
    
            log_file_handle.writeln( str );
        }

        // Sampling Pattern
        location_in_array = getcameravalue("Sampler", currentcamera);
        array = parse(" ", sceneflagvalues[location_in_array]);

        str = "Sampling Pattern: ";

        // Convert sampling type number to corresponding name in UI
        switch( array[2].asNum() )
    	   {
                case 0:
                    str = str + "Blue Noise";
                    break;
    			case 1:
                    str = str + "Fixed";
                    break;
    			case 2:
    				str = str + "Classic";
    				break;
            } // switch

        log_file_handle.writeln( str );


        // Adaptive Sampling
        if( getsceneflagvalue("AdaptiveSampling") != -1 )
        {
            location_in_array = getsceneflagvalue("AdaptiveSampling");
            array = parse(" ", sceneflagvalues[location_in_array]);

            // Is Adaptive Sampling on?
            if(array[2].asNum() == 1)
            {
                // Yes
                log_file_handle.writeln( "Adaptive Sampling: On" );

                // Adamptive Sampling Threshold
                if( getsceneflagvalue("AdaptiveThreshold") != -1 )
                {
                    location_in_array = getsceneflagvalue("AdaptiveThreshold");
                    array = parse(" ", sceneflagvalues[location_in_array]);

                    log_file_handle.writeln( "Adaptive Sampling Threshold: " + array[2].asNum());
                }
            } else
            {
                // No
                log_file_handle.writeln( "Adaptive Sampling: Off" );
            }

        } // Adaptive Sampling

        // Get Camera Oversampling
        location_in_array = getcameravalue("Oversampling", currentcamera);
        array = parse(" ", sceneflagvalues[location_in_array]);
        log_file_handle.writeln( "Oversample: " + array[2].asNum() );

    } else
    {
        // Classic camera AA code here ...

        enhancedaa = FALSE;

        // First we need to know if Enhanced is on (This flag, when turned off, is REMOVED from the scene file)
        if( getsceneflagvalue("EnhancedAA") != -1 ) enhancedaa = TRUE;

        // Antialiasing scene file setting
        if( getsceneflagvalue("AntiAliasing") != -1 )
        {
            location_in_array = getsceneflagvalue("AntiAliasing");
            array = parse(" ", sceneflagvalues[location_in_array]);

            aa = array[2].asNum();
        }

        // Antialiasing Level scene file setting
        if( getsceneflagvalue("AntiAliasingLevel") != -1 )
        {
            location_in_array = getsceneflagvalue("AntiAliasingLevel");
            array = parse(" ", sceneflagvalues[location_in_array]);

            aalevel = array[2].asNum();
        }

        // Get combined value, makes it easier to do a switch statement
        combined_aa = aa * aalevel;

        // Work out Antialiasing setting (as seen in UI)
        // First, is it off completely?
        if( aa == 0 & aalevel == -1 )
        {
            // Yes, Antialiasing is off
            log_file_handle.writeln( "Antialiasing: None" );
        } else
        {
            // Work out which level of AA it is
            switch( combined_aa )
        	   {
                    case -4: // Classic, Extreme
                        str = "Classic, Extreme";
                        if(enhancedaa) str = "Classic, Enhanced Extreme";
                        break;
                    case -3: // Must be Classic AA mode
                        str = "Classic, High";
                        if(enhancedaa) str = "Classic, Enhanced High";
                        break;
                    case -2: // Must be Classic AA mode
                        str = "Classic, Medium";
                        if(enhancedaa) str = "Classic, Enhanced Medium";
                        break;
                    case -1: // Must be Classic AA mode
                        str = "Classic, Low";
                        if(enhancedaa) str = "Classic, Enhanced Low";
                        break;
                    case 0:
                        str = "PLD 01-Pass";
                        break;
                    case 1:
                        str = "PLD 02-Pass";
                        break;
                    case 2:
                        str = "PLD 03-Pass";
                        break;
                    case 3:
                        str = "PLD 04-Pass";
                        break;
                    case 4:
                        str = "PLD 05-Pass";
                        break;
                    case 10:
                        str = "PLD 07-Pass";
                        break;
                    case 12:
                        str = "PLD 09-Pass";
                        break;
                    case 14:
                        str = "PLD 12-Pass";
                        break;
                    case 16:
                        str = "PLD 15-Pass";
                        break;
                    case 27:
                        str = "PLD 17-Pass";
                        break;
                    case 30:
                        str = "PLD 21-Pass";
                        break;
                    case 33:
                        str = "PLD 24-Pass";
                        break;
                    case 36:
                        str = "PLD 28-Pass";
                        break;
                    case 52:
                        str = "PLD 33-Pass";
                        break;
                    case 56:
                        str = "PLD 35-Pass";
                        break;
                } // switch combined_aa

                log_file_handle.writeln( "Antialiasing: " + str );
        } // else


        // Reconstruction Filter
        if( getsceneflagvalue("ReconstructionFilter") != -1 )
        {
            location_in_array = getsceneflagvalue("ReconstructionFilter");
            array = parse(" ", sceneflagvalues[location_in_array]);

            str = "Reconstruction Filter: ";
    
            // Convert sampling type number to corresponding name in UI
            switch( array[2].asNum() )
        	   {
					case 0:
                        str = str + "Classic";
                        break;
        			case 1:
                        str = str + "Box";
                        break;
        			case 3:
        				str = str + "Box (Soft)";
        				break;
        			case 2:
        				str = str + "Box (Sharp)";
        				break;
        			case 4:
        				str = str + "Gaussian";
        				break;
        			case 6:
        				str = str + "Gaussian (Soft)";
        				break;
        			case 5:
        				str = str + "Gaussian (Sharp)";
        				break;
        			case 7:
        				str = str + "Mitchell";
        				break;
        			case 9:
        				str = str + "Mitchell (Soft)";
        				break;
        			case 8:
        				str = str + "Mitchell (Sharp)";
        				break;
        			case 10:
        				str = str + "Lanczos";
        				break;
        			case 12:
        				str = str + "Lanczos (Soft)";
        				break;
        			case 11:
        				str = str + "Lanczos (Sharp)";
        				break;
                } // switch
    
            log_file_handle.writeln( str );
        }

        // Adaptive Sampling
        if( getsceneflagvalue("AdaptiveSampling") != -1 )
        {
            location_in_array = getsceneflagvalue("AdaptiveSampling");
            array = parse(" ", sceneflagvalues[location_in_array]);

            // Is Adaptive Sampling on?
            if(array[2].asNum() == 1)
            {
                // Yes
                log_file_handle.writeln( "Adaptive Sampling: On" );

                // Adamptive Sampling Theshold
                if( getsceneflagvalue("AdaptiveThreshold") != -1 )
                {
                    location_in_array = getsceneflagvalue("AdaptiveThreshold");
                    array = parse(" ", sceneflagvalues[location_in_array]);

                    log_file_handle.writeln( "Theshold: " + array[2].asNum());
                }
            } else
            {
                // No
                log_file_handle.writeln( "Adaptive Sampling: Off" );
            }

        } // Adaptive Sampling

    } // Are we using a Classic camera?






	// Write Global Illumination settings file ...

    interpolated = FALSE;

    // Is radiosity enabled?  If so, look for GI flags
   	if( getsceneflagvalue("EnableRadiosity") != -1 )
   	{
    	// Global Illuminations settings:
    	log_file_handle.writeln( "" );
        if ( formatvb ) log_file_handle.writeln( "[B]Global Illumination:[/B]" );
            else log_file_handle.writeln("Global Illumination:");


    	// Get GI type, find it's location in the sceneflagvalues array, then pull the second value from that location in the array
    	location_in_array = getsceneflagvalue("RadiosityType");

    	array = parse(" ", sceneflagvalues[location_in_array]);

        str = "Radiosity Type: ";

        // Convert radiosity number type to corresponding name in UI
        switch( array[2].asNum() )
    	   {
                case 0:
                    str = str + "Backdrop Only";
                    break;
    			case 1:
                    str = str + "Monte Carlo";
                    break;
    			case 2:
    				str = str + "Final Gather";
    				break;
            } // switch


        // Is it interpolated?  If so, add trailing text to string
    	if( sceneflagvalues[ getsceneflagvalue("RadiosityInterpolated") ] == "RadiosityInterpolated 1" )
    	{
            interpolated = TRUE;
    		str = str + " (Interpolated)";
    	}

        // Write string to file ...
        log_file_handle.writeln( str );

        // Is Blur Radiosity Enabled?
        if( getsceneflagvalue("BlurRadiosity") != -1 )
        {
    	   if( sceneflagvalues[ getsceneflagvalue("BlurRadiosity") ] == "BlurRadiosity 0" )
    	   {
        		log_file_handle.writeln( "Blur Background: Off" );
    	   } else
    	   {
        		log_file_handle.writeln( "Blur Background: On" );
        	}
        }


        // Is Use Transparency Enabled?
        if( getsceneflagvalue("RadiosityTransparency") != -1 )
        {
        	if( sceneflagvalues[ getsceneflagvalue("RadiosityTransparency") ] == "RadiosityTransparency 0" )
        	{
        		log_file_handle.writeln( "Use Transparency: Off" );
        	} else
        	{
        		log_file_handle.writeln( "Use Transparency: On" );
        	}
        }


        // Is Volumetric Radiosity Enabled? (This flag, when turned off, is REMOVED from the scene file)
        if( getsceneflagvalue("VolumetricRadiosity") == -1 )
        {
       		log_file_handle.writeln( "Volumetric Radiosity: Off" );
       	} else
       	{
       		log_file_handle.writeln( "Volumetric Radiosity: On" );
       	}


        // Is Ambient Occulsion Enabled? (This flag, when turned off, is REMOVED from the scene file)
        if( getsceneflagvalue("RadiosityUseAmbient") == -1 )
        {
       		log_file_handle.writeln( "Ambient Occlusion: Off" );
       	} else
       	{
       		log_file_handle.writeln( "Ambient Occlusion: On" );
       	}


        // Is Directional Rays Enabled?
        if( getsceneflagvalue("RadiosityDirectionalRays") != -1 )
        {
    	   if( sceneflagvalues[ getsceneflagvalue("RadiosityDirectionalRays") ] == "RadiosityDirectionalRays 0" )
    	   {
        		log_file_handle.writeln( "Directional Rays: Off" );
    	   } else
    	   {
        		log_file_handle.writeln( "Directional Rays: On" );
        	}
        }

        // These values are greyed out when interpolated is off, so no need to write them out
        if( interpolated == TRUE )
        {
            // Is Use Gradients Enabled?
            if( getsceneflagvalue("RadiosityUseGradients") != -1 )
            {
        	   if( sceneflagvalues[ getsceneflagvalue("RadiosityUseGradients") ] == "RadiosityUseGradients 0" )
        	   {
            		log_file_handle.writeln( "Use Gradients: Off" );
        	   } else
        	   {
            		log_file_handle.writeln( "Use Gradients: On" );
            	}
            }
    
    
            // Is Use Behind Test Enabled?
            if( getsceneflagvalue("RadiosityUseBehindTest") != -1 )
            {
        	   if( sceneflagvalues[ getsceneflagvalue("RadiosityUseBehindTest") ] == "RadiosityUseBehindTest 0" )
        	   {
            		log_file_handle.writeln( "Use Behind Test: Off" );
        	   } else
        	   {
            		log_file_handle.writeln( "Use Behind Test: On" );
            	}
            }
        }

        // Is Use Bumps Enabled?
        if( getsceneflagvalue("RadiosityFlags") != -1 )
        {
            location_in_array = getsceneflagvalue("RadiosityFlags");

            array = parse(" ", sceneflagvalues[location_in_array]);

            // Use Bump Off no RadFlags     = 0
            // Use Bump On  no RadFlags     = -2147483648
            // Use Bump Off with RadFlags   = Some positive number
            // Use Bump On with RadFlags    = Some minus number

            if( array[2].asNum() <0 )
            {
                log_file_handle.writeln( "Use Bumps: On" );
            } else
            {
                log_file_handle.writeln( "Use Bumps: Off" );
            }

        }


        // Radiosity Intensity
        if( getsceneflagvalue("RadiosityIntensity") != -1 )
        {
            location_in_array = getsceneflagvalue("RadiosityIntensity");
            array = parse(" ", sceneflagvalues[location_in_array]);

            log_file_handle.writeln( "Intensity: " + array[2].asNum() * 100 + "%");
        }


        // Indirect Bounces
        if( getsceneflagvalue("IndirectBounces") != -1 )
        {
            location_in_array = getsceneflagvalue("IndirectBounces");
            array = parse(" ", sceneflagvalues[location_in_array]);

            log_file_handle.writeln( "Indirect Bounces: " + array[2].asNum());
        }


        // Rays Per Evaluation
        if( getsceneflagvalue("RadiosityRays") != -1 )
        {
            location_in_array = getsceneflagvalue("RadiosityRays");
            array = parse(" ", sceneflagvalues[location_in_array]);

            log_file_handle.writeln( "Rays Per Evaluation: " + array[2].asNum());
        }


        // These values are greyed out when interpolated is off, so no need to write them out
        if( interpolated == TRUE )
        {
            // Secondary Bounce Rays
            if( getsceneflagvalue("SecondaryBounceRays") != -1 )
            {
                location_in_array = getsceneflagvalue("SecondaryBounceRays");
                array = parse(" ", sceneflagvalues[location_in_array]);

                log_file_handle.writeln( "Secondary Bounce Rays: " + array[2].asNum());
            }
            

            // Angular Tolerance
            if( getsceneflagvalue("RadiosityTolerance") != -1 )
            {
                location_in_array = getsceneflagvalue("RadiosityTolerance");
                array = parse(" ", sceneflagvalues[location_in_array]);

                /*
                    Scene Value = 1.0 - cos(<User Value> * (PI / 180.0))
                    User Value = acos(1.0 - <Scene Value>) * (180.0 / PI)
                */


                AT = array[2].asNum();
                AT = acos(1.0 - AT) * (180.0 / PI);
                log_file_handle.writeln( "Angular Tolerance: " + AT + " Degrees");
            }


            // Minimum Pixel Spacing
            if( getsceneflagvalue("RadiosityMinPixelSpacing") != -1 )
            {
                location_in_array = getsceneflagvalue("RadiosityMinPixelSpacing");
                array = parse(" ", sceneflagvalues[location_in_array]);

                log_file_handle.writeln( "Minimum Pixel Spacing: " + array[2].asNum());
            }


            // Maximum Pixel Spacing
            if( getsceneflagvalue("RadiosityMaxPixelSpacing") != -1 )
            {
                location_in_array = getsceneflagvalue("RadiosityMaxPixelSpacing");
                array = parse(" ", sceneflagvalues[location_in_array]);

                log_file_handle.writeln( "Maximum Pixel Spacing: " + array[2].asNum());
            }


            // Muliplier
            if( getsceneflagvalue("RadiosityMultiplier") != -1 )
            {
                location_in_array = getsceneflagvalue("RadiosityMultiplier");
                array = parse(" ", sceneflagvalues[location_in_array]);

                log_file_handle.writeln( "Multiplier: " + array[2].asNum() * 100 + "%");
            }
            
            cache = FALSE;

            // Is Cache on? (This flag, when turned off, is REMOVED from the scene file)
            if( getsceneflagvalue("CacheRadiosity") == -1 )
            {
           		log_file_handle.writeln( "Cache: Off" );
           	} else
           	{
           	    // Set for easier checking later
                cache = TRUE;
           		log_file_handle.writeln( "Cache: On" );
           	}


           	// Is Cache on? (This flag, when turned off, is REMOVED from the scene file)
           	// Don't write any of the cache values if it is off
            if( cache == TRUE )
            {
                // Get the Preprocess Mode (This flag, when turned off, is REMOVED from the scene file)
                if( getsceneflagvalue("PreprocessRadiosity") != -1 )
                {
                	location_in_array = getsceneflagvalue("PreprocessRadiosity");

                	array = parse(" ", sceneflagvalues[location_in_array]);
            
                    str = "Preprocess Type: ";

                    // Convert radiosity number type to corresponding name in UI
                    switch( array[2].asNum() )
                	   {
                            case 0:
                                str = str + "Automatic";
                                break;
                			case 1:
                                str = str + "Always";
                                break;
                			case 2:
                				str = str + "Never";
                				break;
                			case 3:
                				str = str + "Locked";
                				break;
                        } // switch

                    log_file_handle.writeln( str );
               	}


                // Is Animation on? (This flag, when turned off, is REMOVED from the scene file)
                if( getsceneflagvalue("RadiosityCacheAnimation") == -1 )
                {
               		log_file_handle.writeln( "Animation Mode: Off" );
               	} else
               	{
               		log_file_handle.writeln( "Animation Mode: On" );
               	}
               	

                // Frame Steps
                if( getsceneflagvalue("RadiosityCacheModulus") != -1 )
                {
                    location_in_array = getsceneflagvalue("RadiosityCacheModulus");
                    array = parse(" ", sceneflagvalues[location_in_array]);
    
                    log_file_handle.writeln( "Frame Steps: " + array[2].asNum() );
                }

           	} // if Cache = TRUE

        } // if interpolated = TRUE

   	} // If EnableRadiosity = 1






    // Write out other misc scene flags that might be useful
	log_file_handle.writeln( "" );
	if ( formatvb ) log_file_handle.writeln( "[B]Misc Settings:[/B]" );
        else log_file_handle.writeln("Misc Settings:");

    // Ray Recursion Limit
    if( getsceneflagvalue("RayRecursionLimit") != -1 )
    {
        location_in_array = getsceneflagvalue("RayRecursionLimit");
        array = parse(" ", sceneflagvalues[location_in_array]);

		if ( formatvb ) log_file_handle.writeln( "Ray Recursion Limit: " + array[2].asNum() + "[/SIZE]" );
        	else log_file_handle.writeln( "Ray Recursion Limit: " + array[2].asNum() );
    }

    // All done, close the log file
    log_file_handle.close();
}

